#!/usr/bin/env python3

# Notional Pipe Installer Script
# Copyright (c) 2025, Notional Pipe Incorporated
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of Notional Pipe Incorporated nor the names of its
#       contributors may be used to endorse or promote products derived from this
#       software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

from __future__ import print_function
import six
from six.moves.tkinter import *
import six.moves.tkinter_messagebox as tkMessageBox
from six.moves.tkinter_tkfiledialog import askdirectory
import six.moves.tkinter_font as tkFont
import shutil
import os
import re
import stat
import traceback
import sys
import ctypes
import io
import subprocess
import pipes


def raiseApplication(root=None):
    if root is not None:
        root.lift()


def showErrorDialogBox(title, text):
    raiseApplication()
    tkMessageBox.showwarning(title, text)


def isFrozen():
    return hasattr(sys, "frozen")


def modulePath():
    if isFrozen():
        return os.path.abspath(
            six.ensure_text(sys.executable, sys.getfilesystemencoding())
        )
    else:
        return os.path.dirname(
            os.path.abspath(six.ensure_text(__file__, sys.getfilesystemencoding()))
        )


def windowsUACElevation():
    def isAdmin():
        try:
            return ctypes.windll.shell32.IsUserAnAdmin()
        except:
            return False

    if not isAdmin():
        if isFrozen():
            ctypes.windll.shell32.ShellExecuteW(
                None,
                u"runas",
                six.ensure_text(sys.executable, sys.getfilesystemencoding()),
                six.ensure_text(""),
                None,
                1,
            )
        else:
            ctypes.windll.shell32.ShellExecuteW(
                None,
                u"runas",
                six.ensure_text(sys.executable, sys.getfilesystemencoding()),
                six.ensure_text(__file__, sys.getfilesystemencoding()),
                None,
                1,
            )

        sys.exit(0)


def macOSElevation():
    if os.getuid() == 0:
        return

    scriptPath = os.path.abspath(sys.argv[0])
    escapedScriptPath = "'" + scriptPath.replace("'", "'\"'\"'") + "'"
    escapedScriptPath = escapedScriptPath.replace('"', r"\"")

    appleScript = [
        "/usr/bin/osascript",
        "-e"
        """do shell script "%s" """
        """with prompt "The Contour Rig Tools Installer is trying to install new software." """
        """with administrator privileges""" % (escapedScriptPath,),
    ]
    print(
        "Attempting to re-execute script with administrator privileges: %s"
        % (escapedScriptPath,)
    )
    subprocess.Popen(appleScript, stdin=None)
    sys.exit(0)


def runAsAdministrator():
    if sys.platform == "win32":
        windowsUACElevation()
    elif sys.platform == "darwin":
        macOSElevation()
    elif os.getuid() != 0:  # assuming Linux at this point
        sys.stderr.write("This script must be run as root")
        sys.exit(1)


runAsAdministrator()

root = Tk()
root.withdraw()

moduleDir = os.path.dirname(modulePath())
installerDataPath = os.path.join(moduleDir, "InstallerData")

if not os.path.exists(installerDataPath):
    # In case we're packaged in an app bundle:
    installerDataPath = os.path.join(moduleDir, "Resources", "InstallerData")

    if not os.path.exists(installerDataPath):
        installerDataPath = moduleDir


print("InstallerData path: %s" % (installerDataPath,))

supportedMayaVersions = []

# First installer-info.py must be executed in the global namespace
# in order to get information for this installer like its name and
# supported versions of Maya:

infoPath = os.path.join(installerDataPath, "installer-info.py")

if not os.path.exists(infoPath):
    print("Error: needed file not found: %s" % (infoPath,))
    showErrorDialogBox(
        "Installer Cannot Run",
        "A necessary file for running the installer cannot be found. Make sure this "
        "installer is in the same folder as the 'InstallerData' folder that is "
        "distributed with this installer, or try downloading the installer again. "
        "If you need help, send an email to support@notionalpipe.com that mentions this "
        "error message.",
    )
    raise Exception("Installer error - cannot find installer-info.py")

try:
    root.iconbitmap(default=os.path.join(installerDataPath, "blank.ico"))
except:
    pass

try:
    exec(open(infoPath).read())
except Exception as e:
    showErrorDialogBox(
        "Installer Cannot Run",
        "A necessary file for running the installer is corrupted and cannot be loaded. Try "
        "downloading this installer again. If you need further help, send an email to "
        "support@notionalpipe.com which mentions this error message, and contains the "
        "following diagnostic information that will help solve this problem:\n\n%s"
        % traceback.format_exc(),
    )
    raise e

installerSupportMessage = (
    "If you need help, send an email to support@notionalpipe.com that contains the above error message and describes what steps you've performed so far while trying to install the %s."
    % softwareName
)

installerErrorTitle = softwareName + " Installer Error"
installerDataMissingError = (
    softwareName
    + " Installer Error: Necessary installer data is missing.\n\nMake sure the installer is in the same folder as the 'InstallerData' folder that is distributed with this installer, or try downloading the "
    + softwareName
    + " again.\n\nCannot continue with the installation.\n\n"
    + installerSupportMessage
)
installerMissingMayaDirectory = (
    softwareName
    + " Installer Error: Couldn't locate the %s directory for the Maya installation located at '%s'.\n\nCannot continue with the installation.\n\n"
    + installerSupportMessage
)
installerFileBadPermissions = (
    softwareName
    + " Installer Error: The following files on disk need to be overwritten "
    "by the installer, but do not have write permission or otherwise aren't allowing "
    "themselves to be overwritten:\n\n%s\n\n"
    + softwareName
    + " cannot be installed.\n\n"
    + "Please check the permissions on these files and quit Maya and any other software that "
    "may be using them.\n\n" + installerSupportMessage
)
installerInstallationFailed = (
    softwareName
    + " Installer Error: One or more files failed to copy to their target directory.\n\n%s\n\nWARNING: "
    + softwareName
    + " may now be partially installed but incomplete.  It is not advisable to try and use it.  First resolve this error, and then try running the installer again.\n\n"
    + installerSupportMessage
)
installerOtherError = (
    softwareName
    + " Installer Error: An unexpected error occurred!  Here is the error:\n\n%s\n\nWARNING: If you began installing "
    + softwareName
    + ", it may now be partially installed but incomplete.  It is not advisable to try and use it.  First resolve this error, and then try running the installer again.\n\n"
    + installerSupportMessage
)
ignoreFiles = [".DS_Store"]


def displayErrors(f):
    """Decorator that catches all exceptions and displays them to the user, so they don't get silently swallowed."""

    def new_f(*args, **kw):
        try:
            return f(*args, **kw)
        except Exception as e:
            print(traceback.format_exc())
            showErrorDialogBox(
                installerErrorTitle, installerOtherError % traceback.format_exc()
            )

    return new_f


class InvalidFolderDeletionError(Exception):
    pass


class UnsupportedMayaVersionError(Exception):
    pass


def joinListToPath(path, items):
    """Adds all of the strings in list 'items' on to the path 'path'"""
    for item in items:
        path = os.path.join(path, item)
    return path


def isMayaDirectory(path):
    """Checks if path points to a valid installation of Maya by checking for the Maya executible"""
    contents = os.listdir(path)
    if "Maya.app" in contents:
        return True
    elif "bin" in contents:
        contents2 = os.listdir(os.path.join(path, "bin"))
        contents2 = [file.lower() for file in contents2]
        if "maya" in contents2 or "maya.exe" in contents2 or "maya.bin" in contents2:
            return True
    return False


def mayaVersionFromPath(path):
    """Determines the main Maya version from a path to a valid Maya installation, and returns it as a string.
    The version can include a decimal (e.g. the path to Maya 2013 extension will return "2013.5"). Returns the
    string '(Unknown version)' if the Maya version cannot be determined."""
    dir = os.path.basename(path)
    if not dir:
        dir = path.split(os.path.sep)[-2]

    versionString = "(Unknown version)"
    match = re.search(
        "(aya[ ]*)([0-9.]+)", dir
    )  # matches the 2011 in "maya2011", "maya 2011", "Maya 2011", "maya2011ext2", etc.
    if match:
        versionString = match.group(2)
    return versionString


def mayaVersionNumberFromPath(dirPath):
    """Returns the Maya version number from a path to a valid Maya installation as a float. Returns 0 if
    dirPath is not the path to a Maya directory."""
    versionString = mayaVersionFromPath(dirPath)
    try:
        versionNumber = float(versionString)
    except:
        versionNumber = 0
    return versionNumber


def mayaVersionDisplayStringFromPath(path):
    """Returns a displayable string for the Maya version specified by path."""
    # This function almost does the same thing as mayaVersionFromPath, except that it will
    # return "2013 Extension" instead of "2013.5" for Maya 2013 Extension.
    result = mayaVersionFromPath(path)
    if result == "2013.5":
        result = "2013 Extension"
    return result


def versionIsSupported(versionNumber):
    return versionNumber in supportedMayaVersions


def findMayaInstallations():
    """Returns a list of paths to current Maya installations that meet the requirements of this installer (version >= 2011)"""
    if os.name == "nt":
        mayaSearchPaths = []
        for var in ["ProgramFiles", "ProgramFiles(x86)", "ProgramW6432"]:
            if var in os.environ and len(os.environ[var]) > 0:
                path = os.environ[var] + "\\Autodesk\\"
                if path not in mayaSearchPaths:
                    mayaSearchPaths.append(os.environ[var] + "\\Autodesk\\")
    else:
        mayaSearchPaths = ["/Applications/Autodesk/", "/autodesk/", "/usr/autodesk/"]

    mayaInstallations = []
    for path in mayaSearchPaths:
        if os.path.exists(path):
            dirs = os.listdir(path)

            for casedDir in dirs:
                dir = casedDir.lower()
                dirPath = os.path.join(path, casedDir)

                if dir.startswith("maya") and isMayaDirectory(dirPath):
                    versionNumber = mayaVersionNumberFromPath(dirPath)

                    if versionIsSupported(versionNumber):
                        mayaInstall = path + casedDir

                        if mayaInstall not in mayaInstallations:
                            mayaInstallations.append(mayaInstall)

    return sorted(mayaInstallations)


def findSitePackages(mayaDirectory, mayaVersion, pythonVersion):
    """Returns a list of the site-packages directory for the Maya installation at mayaDirectory,
    if it exists, regardless of operating system"""

    def findDirectory(start, items):
        path = start
        for item in items:
            if not os.path.exists(path):
                break
            itemToAdd = item
            if item.endswith("*"):
                dirs = os.listdir(path)
                for dir in dirs:
                    if dir.startswith(item[:-1]) and os.path.isdir(
                        os.path.join(path, dir)
                    ):
                        itemToAdd = dir
                        break
            path = os.path.join(path, itemToAdd)

        if os.path.exists(path):
            return path
        else:
            return None

    if sys.platform == "linux" or sys.platform == "linux2":
        path = ["lib", "python%s" % pythonVersion, "site-packages"]
    elif sys.platform == "darwin":
        path = [
            "Maya.app",
            "Contents",
            "Frameworks",
            "Python.framework",
            "Versions",
            "Current",
            "lib",
            "python*",
            "site-packages",
        ]
    elif sys.platform == "win32":
        if mayaVersion == 2022:
            path = ["Python%s" % pythonVersion.replace(".", ""), "lib", "site-packages"]
        else:
            path = ["Python", "lib", "site-packages"]

    directory = findDirectory(mayaDirectory, path)

    if directory is None:
        raise Exception(
            "Failed to find Python %s site-packages directory for Maya %s"
            % (pythonVersion, mayaVersion)
        )

    return directory


def findPluginsDirectory(mayaDirectory):
    """Returns the plug-ins directory for the Maya installation at mayaDirectory.
    NOTE: In Mac OS X this will create the plug-ins directory if it doesn't exist,
    since it's not guaranteed to exist for a valid Maya installation."""

    if sys.platform == "darwin":
        mayaVersion = mayaVersionFromPath(mayaDirectory)
        if mayaVersion != "(Unknown version)":
            path = joinListToPath(
                "/Users/Shared/Autodesk/maya", [mayaVersion, "plug-ins"]
            )

            if not os.path.exists(path):
                try:
                    os.makedirs(path)
                except OSError:
                    path = None
                    pass

        else:
            path = None

    else:
        path = joinListToPath(mayaDirectory, ["bin", "plug-ins"])

    return path


def findIconsDirectory(mayaDirectory):
    """Returns the plug-ins directory for the Maya installation at mayaDirectory
    NOTE: In Mac OS X this will create the icons directory if it doesn't exist, since
    it's not guaranteed to exist for a valid Maya installation."""

    if sys.platform == "darwin":
        mayaVersion = mayaVersionFromPath(mayaDirectory)
        if mayaVersion != "(Unknown version)":
            path = joinListToPath("/Users/Shared/Autodesk/maya", [mayaVersion, "icons"])

            if not os.path.exists(path):
                try:
                    os.makedirs(path)
                except OSError:
                    path = None
                    pass
        else:
            path = None

    else:
        path = joinListToPath(mayaDirectory, ["icons"])

    return path


def displayMissingFilesError():
    showErrorDialogBox(installerErrorTitle, installerDataMissingError)


def mayaPythonVersions(mayaVersion):
    """Returns the versions of Python included with the given version of Maya as a list of strings,
    in the format major.minor. (e.g. 2.7)"""
    if mayaVersion == 2025:
        return ["3.11"]

    if mayaVersion == 2024:
        return ["3.10"]

    if mayaVersion == 2023:
        return ["3.9"]

    if mayaVersion == 2022:
        return ["2.7", "3.7"]

    if mayaVersion >= 2014 and mayaVersion < 2022:
        return ["2.7"]

    if mayaVersion >= 2010 and mayaVersion < 2014:
        return ["2.6"]

    # We don't care about the Python version for versions of Maya earlier than 2010. In fact we don't
    # really care about 2010 either but it's included in here just to be a little extra thorough.
    raise UnsupportedMayaVersionError


def mayaVersionString(mayaVersion):
    """ Returns a string representation of the Maya version"""
    if int(mayaVersion) == mayaVersion:
        return str(int(mayaVersion))
    else:
        return str(mayaVersion)


def getSubdirectoryFiles(subdirectory, versionSpecificDir=None):
    """Returns list of paths to files in subdirectory for the given version specific directory. If no version
    specific subdirectory is provided, returns only files that are in the general subdirectory (i.e. will
    not include files contained in 'InstallerData/2.6/site-packages', but will return files in
    'InstallerData/site-packages'). Returns empty list if no files are found or if the subdirectory does
    not exist."""
    if versionSpecificDir is None:
        subdirPath = os.path.join(installerDataPath, subdirectory)
    else:
        subdirPath = os.path.join(installerDataPath, versionSpecificDir, subdirectory)

    if os.path.exists(subdirPath):
        return [os.path.join(subdirPath, file) for file in os.listdir(subdirPath)]
    else:
        return []


def getFilesToInstall(subdirectory, mayaVersion, pythonVersion=None):
    """Returns list of paths to files in subdirectory that are appropriate for the given Maya version. Returns empty list
    if no files are found."""
    result = getSubdirectoryFiles(subdirectory, None) + getSubdirectoryFiles(
        subdirectory, mayaVersionString(mayaVersion)
    )

    if pythonVersion is not None:
        result += getSubdirectoryFiles(subdirectory, pythonVersion)

    return result


def ensureReadable(file):
    currentPermissions = stat.S_IMODE(os.stat(file).st_mode)
    os.chmod(file, currentPermissions | stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH)


def allInstallationFilesForTargetRoot(srcDir, targetRoot):
    return [
        (
            root,
            os.path.join(targetRoot, root[len(os.path.dirname(srcDir)) + 1 :]),
            files,
        )
        for root, dirs, files in os.walk(srcDir)
    ]


def copyFileToTarget(file, targetDir):
    if os.path.basename(file) in ignoreFiles:
        return

    targetInstallationPath = os.path.join(targetDir, os.path.basename(file))
    shutil.copy(file, targetDir)
    resultantFile = os.path.join(targetDir, os.path.basename(file))
    ensureReadable(resultantFile)


def recursiveCopy(srcDir, targetRoot):
    for root, currentDir, files in allInstallationFilesForTargetRoot(
        srcDir, targetRoot
    ):
        if not os.path.exists(currentDir):
            os.makedirs(currentDir)
            ensureReadable(currentDir)

        for file in files:
            copyFileToTarget(os.path.join(root, file), currentDir)


def copyDirectoryToTarget(directory, targetRoot):
    basename = os.path.basename(directory)
    targetInstallationPath = os.path.join(targetRoot, basename)

    if os.path.exists(targetInstallationPath):
        if basename not in deletableDirectories:
            raise InvalidFolderDeletionError(
                "Not allowed to delete folder: %s" % targetInstallationPath
            )

        shutil.rmtree(targetInstallationPath)

    recursiveCopy(directory, targetRoot)


def installFiles(files, targetRoot):
    """Installs files to targetRoot"""
    for file in files:
        if os.path.isdir(file):
            copyDirectoryToTarget(file, targetRoot)
        else:
            copyFileToTarget(file, targetRoot)


def checkExistenceOfInstallationFiles():
    """Returns true if the expected installation files are found, otherwise returns false."""
    for subdir in requiredInstallationDirectories:
        files = getSubdirectoryFiles(subdir, None)

        for pythonVersion in supportedMayaPythonVersions:
            files += getSubdirectoryFiles(subdir, pythonVersion)

        for mayaVersion in supportedMayaVersions:
            files += getSubdirectoryFiles(subdir, mayaVersionString(mayaVersion))

        if len(files) == 0:
            print("Error: needed files were not found for: %s" % (subdir,))
            return False

    return True


def getInstallationFilesAndTargets(mayaDirectories):
    """Returns a dict containing dicts.  Outer dict has keys for each maya directory.  Inner dict has keys for each
    directory that will have files installed into it, and the value is a list of the files to be installed in that
    directory.  In case of a known error, a dialog will be displayed to the user telling them what went wrong, and
    the function will return nothing."""
    result = {}
    for mayaDirectory in mayaDirectories:
        mayaVersion = mayaVersionNumberFromPath(mayaDirectory)
        pluginsFiles = getFilesToInstall("plug-ins", mayaVersion)
        iconsFiles = getFilesToInstall("icons", mayaVersion)

        mayaInstallData = {}

        pluginsPath = findPluginsDirectory(mayaDirectory)
        if pluginsPath is None or not os.path.exists(pluginsPath):
            showErrorDialogBox(
                installerErrorTitle,
                installerMissingMayaDirectory % ("plug-ins", mayaDirectory),
            )
            return

        iconsPath = findIconsDirectory(mayaDirectory)
        if iconsPath is None or not os.path.exists(iconsPath):
            showErrorDialogBox(
                installerErrorTitle,
                installerMissingMayaDirectory % ("icons", mayaDirectory),
            )
            return

        pythonVersions = mayaPythonVersions(mayaVersion)

        for pythonVersion in pythonVersions:
            sitePackagesPath = findSitePackages(
                mayaDirectory, mayaVersion, pythonVersion
            )

            if sitePackagesPath is None or not os.path.exists(sitePackagesPath):
                showErrorDialogBox(
                    installerErrorTitle,
                    installerMissingMayaDirectory
                    % (("Python %s site-packages" % pythonVersion), mayaDirectory),
                )
                return

            sitePackagesFiles = getFilesToInstall(
                "site-packages", mayaVersion, pythonVersion=pythonVersion
            )

            mayaInstallData[sitePackagesPath] = sitePackagesFiles

        mayaInstallData[pluginsPath] = pluginsFiles
        mayaInstallData[iconsPath] = iconsFiles

        result[mayaDirectory] = mayaInstallData

    return result


def pathHasWriteAccess(path):
    if not os.access(path, os.W_OK):
        return False

    if os.path.isdir(path):
        return True

    try:
        with open(path, "a") as f:
            return True
    except IOError:
        return False


def uniqueTemporaryName(path):
    def tempFilePath(dir, basename, count):
        return os.path.join(dir, basename + "_temp_" + str(count))

    dir = os.path.dirname(path)
    basename = os.path.basename(path)
    count = 1

    while os.path.exists(tempFilePath(dir, basename, count)):
        count += 1

        if count > 9:
            raise Exception("Too many temporary directories")

    return tempFilePath(dir, basename, count)


def filesWithoutWritePermission(allInstallData):
    """Checks to make sure that all of the target directories in allInstallData have write access. Returns
    a list of directories with bad permissions if it finds any, otherwise an empty list."""
    result = []

    def checkFolderCanBeModified(path):
        tempDirectory = uniqueTemporaryName(path)

        try:
            os.rename(path, tempDirectory)
            os.rename(tempDirectory, path)
        except:
            print("Cannot modify %s" % path)
            result.append(path)

    def checkFileWriteAccess(targetRoot, file):
        targetFile = os.path.join(targetRoot, os.path.basename(file))

        if os.path.exists(targetFile) and not pathHasWriteAccess(targetFile):
            print(
                "Cannot install over",
                targetFile,
                "; do not have write access for that file",
            )
            result.append(targetFile)

    def recursivelyCheckDirectoryWriteAccess(targetRoot, directory):
        for root, currentDir, files in allInstallationFilesForTargetRoot(
            directory, targetRoot
        ):
            if os.path.exists(currentDir) and not pathHasWriteAccess(currentDir):
                print(
                    "Cannot install to",
                    currentDir,
                    "; do not have write access to that directory",
                )
                result.append(currentDir)

            for file in files:
                checkFileWriteAccess(currentDir, file)

        targetDirectory = os.path.join(targetRoot, os.path.basename(directory))

        if os.path.exists(targetDirectory):
            checkFolderCanBeModified(targetDirectory)

    for mayaDirectory, installData in allInstallData.items():
        for targetRoot, installerFiles in installData.items():
            for installerFile in installerFiles:
                if os.path.isdir(installerFile):
                    recursivelyCheckDirectoryWriteAccess(targetRoot, installerFile)
                else:
                    checkFileWriteAccess(targetRoot, installerFile)

    return result


def doInstall(mayaDirectories):
    """Finds files to be installed for each Maya installation in mayaDirectories, checks to make sure permissions are
    okay, and then installs the files.  If something goes wrong, displays an error dialog and aborts the installation."""

    allInstallData = getInstallationFilesAndTargets(mayaDirectories)
    if allInstallData is None:
        return False

    problemFiles = filesWithoutWritePermission(allInstallData)

    if len(problemFiles) > 0:
        if len(problemFiles) > 5:
            problemFiles = problemFiles[:5] + ["[ more files not displayed ]"]
        showErrorDialogBox(
            installerErrorTitle,
            installerFileBadPermissions % ("\n\n".join(problemFiles)),
        )
        return False

    try:
        for mayaDirectory, installData in allInstallData.items():
            for targetRoot, files in installData.items():
                installFiles(files, targetRoot)
    except IOError as e:
        print(traceback.format_exc())
        showErrorDialogBox(installerErrorTitle, installerInstallationFailed % str(e))
        return False

    return True


def getLicense():
    """Returns the text of the EULA for this installer"""
    licensePath = os.path.join(installerDataPath, "license.txt")
    try:
        file = io.open(licensePath, "r", encoding="utf8")
        licenseText = file.read()
        file.close()
        licenseText = licenseText.replace("\r", "")  # Remove windows line breaks
        return licenseText
    except IOError:
        print("Warning: Could not read license.txt")
        return "License agreement not found"


def quitApplication():
    """Does what it says"""
    global root
    root.destroy()


# Maya installer window implemented this way so that we can poll the state of the list of Maya installations
class Dialog(Frame):
    def __init__(self, master):
        Frame.__init__(self, master)

        self.mayaListPaths = []

        self.label1 = Label(master, text=(softwareName + " installer"), pady=10)
        self.label1.pack(side=TOP)

        self.label2 = Label(
            master,
            text=(
                "Please select the Maya installations for which you want to install "
                + softwareName
                + ":"
            ),
            justify=LEFT,
            wraplength=500,
        )
        self.label2.pack(side=TOP, fill=BOTH)

        self.listFrame = Frame(master)
        self.listFrame.pack(side=TOP, fill=BOTH, expand=1)

        scrollbarY = Scrollbar(self.listFrame, orient=VERTICAL)
        scrollbarX = Scrollbar(self.listFrame, orient=HORIZONTAL)
        self.mayaList = Listbox(
            self.listFrame,
            selectmode=MULTIPLE,
            yscrollcommand=scrollbarY.set,
            xscrollcommand=scrollbarX.set,
        )

        scrollbarY.config(command=self.mayaList.yview)
        scrollbarY.pack(side=RIGHT, fill=Y)
        scrollbarX.config(command=self.mayaList.xview)
        scrollbarX.pack(side=BOTTOM, fill=X)

        self.populateMayaList()
        self.mayaList.pack(side=TOP, fill=BOTH, expand=1)
        self.mayaListCurrent = None

        self.buttonFrame = Frame(master)
        self.buttonFrame.pack(side=BOTTOM, fill=X)

        self.installButton = Button(
            self.buttonFrame,
            text="Install",
            state=DISABLED,
            default=ACTIVE,
            command=self.install,
        )
        self.installButton.pack(side=RIGHT)

        self.otherButton = Button(
            self.buttonFrame,
            text="Add Other Maya Installation...",
            command=self.addOtherMayaInstall,
        )
        self.otherButton.pack(side=LEFT)

        self.poll()  # start polling the list

    def addMayaInstallationToList(self, directory):
        self.mayaList.insert(
            END,
            "Maya " + mayaVersionDisplayStringFromPath(directory) + ": " + directory,
        )
        self.mayaListPaths.append(directory)

    def populateMayaList(self):
        mayaLocations = findMayaInstallations()
        for mayaLocation in mayaLocations:
            self.addMayaInstallationToList(mayaLocation)

    def addOtherMayaInstall(self):
        directory = askdirectory()

        if len(directory) == 0:
            return

        if not isMayaDirectory(directory):
            showErrorDialogBox(
                installerErrorTitle,
                "The directory:\n%s\nis not a Maya installation." % directory,
            )
            return

        self.addMayaInstallationToList(directory)

    def poll(self):
        now = self.mayaList.curselection()
        try:
            now = list(map(int, now))
        except ValueError:
            pass
        if now != self.mayaListCurrent:
            self.mayaListSelectionChanged(now)
            self.mayaListCurrent = now
        self.after(100, self.poll)

    def mayaListSelectionChanged(self, selection):
        if len(selection) > 0:
            self.installButton.config(state=NORMAL)
        else:
            self.installButton.config(state=DISABLED)

    @displayErrors
    def install(self):
        self.installButton.config(text="Installing...")
        self.installButton.config(state=DISABLED)
        root.update_idletasks()
        try:
            result = doInstall([self.mayaListPaths[i] for i in self.mayaListCurrent])
            if result == True:
                global successWindowTopLevel
                global mayaWindowTopLevel
                mayaWindowTopLevel.withdraw()
                successWindowTopLevel.deiconify()
        finally:
            self.installButton.config(text="Install")
            self.installButton.config(state=NORMAL)


class MayaWindow(Frame):
    @displayErrors
    def __init__(self, master):
        paddingFrame = Frame(master)
        paddingFrame.pack(padx=10, pady=10, fill=BOTH, expand=1)

        frame = Dialog(paddingFrame)
        frame.pack(fill=BOTH, expand=1)


class EulaWindow(Frame):
    def nextWindow(self):
        global eulaWindowTopLevel
        global mayaWindowTopLevel
        eulaWindowTopLevel.withdraw()
        mayaWindowTopLevel.deiconify()

    def acceptButtonPressed(self):
        if self.checkBoxState.get() != 0:
            self.eulaNextButton.config(state=NORMAL)
        else:
            self.eulaNextButton.config(state=DISABLED)

    @displayErrors
    def __init__(self, master):

        self.topLevel = master

        paddingFrame = Frame(master)
        paddingFrame.pack(padx=10, pady=10, fill=BOTH, expand=1)

        label1 = Label(
            paddingFrame,
            text=("Please read the license agreement for " + softwareName + ":"),
            anchor=W,
        )
        label1.pack(side=TOP)

        listFrame = Frame(paddingFrame, borderwidth=2, relief=RIDGE)
        listFrame.pack(side=TOP, fill=BOTH, expand=1)

        scrollbarY = Scrollbar(listFrame, orient=VERTICAL)
        eulaTextField = Text(
            listFrame, borderwidth=0, wrap=WORD, yscrollcommand=scrollbarY.set
        )
        scrollbarY.config(command=eulaTextField.yview)

        eulaTextField.pack(side=LEFT, fill=BOTH, expand=1)
        scrollbarY.pack(side=RIGHT, fill=Y)

        eulaTextField.insert(END, getLicense())

        buttonFrame = Frame(paddingFrame)
        buttonFrame.pack(side=TOP, fill=X)

        self.checkBoxState = IntVar()
        acceptButton = Checkbutton(
            buttonFrame,
            text="I accept the terms of this license agreement.",
            variable=self.checkBoxState,
            command=self.acceptButtonPressed,
        )
        acceptButton.pack(side=LEFT)

        self.eulaNextButton = Button(
            buttonFrame,
            text="Next",
            state=DISABLED,
            default=ACTIVE,
            command=self.nextWindow,
        )
        self.eulaNextButton.pack(side=RIGHT)


class SuccessWindow(Frame):
    @displayErrors
    def __init__(self, master):
        paddingFrame = Frame(master)
        paddingFrame.pack(padx=10, pady=10, fill=BOTH, expand=1)

        label = Label(
            paddingFrame,
            text="The installation completed successfully!",
            anchor=W,
            pady=10,
        )
        label.pack(side=TOP)

        boldFont = tkFont.Font(font=label["font"])
        boldFont.config(weight="bold")
        label.config(font=boldFont)

        if sys.platform.startswith("darwin"):
            label = Label(
                paddingFrame,
                text=installerReadMeMessage,
                anchor=W,
                justify=LEFT,
                wraplength=500,
                font=("system", 12),
            )
        else:
            label = Label(
                paddingFrame,
                text=installerReadMeMessage,
                anchor=W,
                justify=LEFT,
                wraplength=500,
            )
        label.pack(side=TOP)

        button = Button(
            paddingFrame, text="Quit", default=ACTIVE, command=quitApplication
        )
        button.pack(side=TOP)


if not checkExistenceOfInstallationFiles():
    displayMissingFilesError()
else:
    mayaWindowTopLevel = Toplevel()
    mayaWindowTopLevel.protocol("WM_DELETE_WINDOW", quitApplication)
    mayaWindow = MayaWindow(mayaWindowTopLevel)
    mayaWindowTopLevel.withdraw()
    mayaWindowTopLevel.title(softwareName + " Installer")

    eulaWindowTopLevel = Toplevel()
    eulaWindowTopLevel.protocol("WM_DELETE_WINDOW", quitApplication)
    eulaWindow = EulaWindow(eulaWindowTopLevel)
    eulaWindowTopLevel.withdraw()
    eulaWindowTopLevel.title(softwareName + " Installer")

    successWindowTopLevel = Toplevel()
    successWindowTopLevel.protocol("WM_DELETE_WINDOW", quitApplication)
    successWindow = SuccessWindow(successWindowTopLevel)
    successWindowTopLevel.withdraw()
    successWindowTopLevel.title(softwareName + " Installer")

    eulaWindowTopLevel.deiconify()
    raiseApplication(root)

    root.mainloop()
