import sys
import os
import maya.cmds as cmds
import subprocess
import inspect
import platform
import traceback
import site
from Contourlib.mayaversions.mayaversions import isPySide6

if isPySide6:
    from PySide6 import QtWidgets, QtCore, QtGui
    from PySide6.QtWidgets import (
        QApplication,
        QWidget,
        QMessageBox,
        QDialog,
        QVBoxLayout,
        QLabel,
        QPlainTextEdit,
        QPushButton,
        QDialogButtonBox,
        QProgressBar,
        QStyleFactory,
    )
    from PySide6.QtCore import QObject, QProcess
else:
    from PySide2 import QtWidgets, QtCore, QtGui
    from PySide2.QtWidgets import (
        QApplication,
        QWidget,
        QMessageBox,
        QDialog,
        QVBoxLayout,
        QLabel,
        QPlainTextEdit,
        QPushButton,
        QDialogButtonBox,
        QProgressBar,
        QStyleFactory,
    )
    from PySide2.QtCore import QObject, QProcess
import Contourlib.contoursix as six


def mayaVersion():
    return cmds.about(apiVersion=True) // 10000


def pymelDocURL():
    mayaVer = mayaVersion()

    if mayaVer == 2024:
        return "https://help.autodesk.com/view/MAYAUL/2024/ENU/?guid=GUID-2AA5EFCE-53B1-46A0-8E43-4CD0B2C72FB4"
    if mayaVer == 2023:
        return "https://help.autodesk.com/view/MAYAUL/2023/ENU/?guid=GUID-2AA5EFCE-53B1-46A0-8E43-4CD0B2C72FB4"
    elif mayaVer == 2022:
        return "https://help.autodesk.com/view/MAYAUL/2022/ENU/?guid=GUID-2AA5EFCE-53B1-46A0-8E43-4CD0B2C72FB4"

    return "https://help.autodesk.com/view/MAYAUL/2025/ENU/?guid=GUID-2AA5EFCE-53B1-46A0-8E43-4CD0B2C72FB4"


def pymelWhlPath():
    import Contourlib.pymelinstaller

    if mayaVersion() >= 2025:
        whlPath = "pymel-1.5.0-py2.py3-none-any.whl"
    else:
        whlPath = "pymel-1.4.0-py2.py3-none-any.whl"

    return os.path.join(
        os.path.dirname(inspect.getfile(Contourlib.pymelinstaller)), whlPath,
    )


def mayapyPath():
    if platform.system() == "Windows":
        path = os.path.join(os.path.dirname(sys.executable), "mayapy.exe")
    elif platform.system() == "Darwin":
        path = os.path.join(
            os.path.dirname(os.path.dirname(sys.executable)), "bin", "mayapy"
        )
    else:
        return None

    if os.path.exists(path) and os.access(path, os.X_OK):
        return path

    return None


def promptUserForPyMELInstall():
    messageBox = QMessageBox()
    messageBox.setIcon(QMessageBox.Question)
    messageBox.setTextFormat(QtCore.Qt.RichText)
    messageBox.setText("<h3>PyMEL must be installed to run Contour</h3>")
    messageBox.setInformativeText(
        "<p>We can install PyMEL for you, or you can install it manually by following the instructions in "
        '<a href="%s">this document</a> from Autodesk Support.</p>' % pymelDocURL()
    )
    noButton = messageBox.addButton("Cancel", QMessageBox.NoRole)
    yesButton = messageBox.addButton("Install PyMEL", QMessageBox.YesRole)
    messageBox.setDefaultButton(noButton)
    messageBox.exec_()

    return messageBox.clickedButton() == yesButton


def displayInstallSucceeded():
    messageBox = QMessageBox()
    messageBox.setIcon(QMessageBox.Information)
    messageBox.setText("PyMEL was successfully installed")
    messageBox.exec_()


def displayInstallCanceled():
    messageBox = QMessageBox()
    messageBox.setIcon(QMessageBox.Warning)
    messageBox.setText("PyMEL installation was canceled")
    messageBox.exec_()


def displayGeneralFailure(message):
    messageBox = QMessageBox()
    messageBox.setIcon(QMessageBox.Critical)
    messageBox.setTextFormat(QtCore.Qt.RichText)
    messageBox.setText("<h3>Failed to Install PyMEL</h3>")
    messageBox.setInformativeText(
        "<p>%s</p>"
        "<p>You can install PyMEL manually by following the instructions in "
        '<a href="%s">this document</a> from Autodesk Support.</p>'
        "<p>If you need assistance, contact Notional Pipe support at "
        '<a href="mailto:support@notionalpipe.com">support@notionalpipe.com</a></p>'
        % (message, pymelDocURL())
    )
    messageBox.exec_()


def displayFailedWithOutput(message, output):
    dialog = QDialog()
    dialog.setMinimumWidth(480)
    layout = QVBoxLayout(dialog)

    header = QLabel('<h3 style="text-align: center">%s</h3>' % message, dialog)
    header.setTextFormat(QtCore.Qt.RichText)

    text1 = QLabel(
        "<p>You can install PyMEL manually by following the instructions in "
        '<a href="%s">this document</a> from Autodesk Support.</p>' % pymelDocURL(),
        dialog,
    )
    text1.setWordWrap(True)
    text1.setTextFormat(QtCore.Qt.RichText)
    text1.setTextInteractionFlags(QtCore.Qt.TextBrowserInteraction)
    text1.setOpenExternalLinks(True)

    text2 = QLabel(
        "<p>If you need assistance, contact Notional Pipe support at "
        '<a href="mailto:support@notionalpipe.com">support@notionalpipe.com</a><br/>'
        "Please include the following text in your email:</p>",
        dialog,
    )
    text2.setWordWrap(False)
    text2.setTextFormat(QtCore.Qt.RichText)
    text2.setTextInteractionFlags(QtCore.Qt.TextBrowserInteraction)
    text2.setOpenExternalLinks(True)

    okButton = QDialogButtonBox()
    okButton.addButton(QDialogButtonBox.Ok)
    okButton.accepted.connect(dialog.accept)

    outputBox = QPlainTextEdit(dialog)
    outputBox.setPlainText(output)
    outputBox.setReadOnly(True)
    outputBox.setWordWrapMode(QtGui.QTextOption.NoWrap)
    outputBox.setFont(QtGui.QFontDatabase.systemFont(QtGui.QFontDatabase.FixedFont))

    layout.addWidget(header)
    layout.addWidget(text1)
    layout.addWidget(text2)
    layout.addWidget(outputBox)
    layout.addWidget(okButton)

    dialog.resize(dialog.sizeHint().width(), 250)

    dialog.exec_()


def displayPymelManualInstallInstructions():
    messageBox = QMessageBox()
    messageBox.setIcon(QMessageBox.Information)
    messageBox.setTextFormat(QtCore.Qt.RichText)
    messageBox.setText("<h3>PyMEL must be installed to run Contour</h3>")
    messageBox.setInformativeText(
        "<p><b>To install PyMEL, "
        'see <a href="%s">these instructions</a> from Autodesk Support.</b></p>'
        "<p>If you need assistance, please contact Notional Pipe support at "
        '<a href="mailto:support@notionalpipe.com">support@notionalpipe.com</a></p>'
        % pymelDocURL()
    )
    messageBox.exec_()


def installPyMEL(mayapy, pymelWhl):
    print("Installing PyMEL using pip...")
    print("-----------------------------")

    p = QProcess()
    p.setProgram(mayapy)

    args = [
        "-m",
        "pip",
        "install",
        "--no-python-version-warning",
        "--disable-pip-version-check",
        "--no-input",
        "--no-color",
        pymelWhl,
    ]

    p.setArguments(args)
    p.setProcessChannelMode(QProcess.MergedChannels)

    progressWindow = QDialog()
    progressWindow.setWindowFlags(
        QtCore.Qt.CustomizeWindowHint | QtCore.Qt.WindowTitleHint
    )
    progressWindow.setModal(True)
    layout = QVBoxLayout(progressWindow)

    label = QLabel("Installing PyMEL...")

    progress = QProgressBar()
    progress.setRange(0, 0)
    progress.setStyle(QStyleFactory.create("Fusion"))

    buttonBox = QDialogButtonBox(progressWindow)
    buttonBox.setStandardButtons(QDialogButtonBox.Cancel)

    layout.addWidget(label)
    layout.addWidget(progress)
    layout.addWidget(buttonBox)
    progressWindow.setLayout(layout)

    data = {"canceled": False, "output": ""}

    def cancel():
        label.setText("Canceling...")
        buttonBox.setEnabled(False)
        data["canceled"] = True

        if platform.system() == "Windows":
            p.kill()
        else:
            p.terminate()

    def readOutput():
        bytes = p.readAllStandardOutput().data()
        sys.stdout.write(bytes)
        data["output"] += six.ensure_text(bytes)

    p.finished.connect(progressWindow.close)
    buttonBox.rejected.connect(cancel)
    p.readyReadStandardOutput.connect(readOutput)

    p.start()
    progressWindow.exec_()

    readOutput()
    print("-----------------------------")

    if p.exitStatus() != QProcess.CrashExit:
        print("pip exit code: %s" % p.exitCode())
        succeeded = p.exitCode() == 0
    else:
        print("pip terminated without status code")
        succeeded = False

    print("Canceled" if data["canceled"] else "Finished")

    return (
        succeeded,
        data["output"],
        data["canceled"],
    )


def ensureUserSitePackagesAdded():
    path = site.getusersitepackages()

    if path is None:
        print(
            "Warning: site.getusersitepackages() returned None, can't determine user site packages directory"
        )
        return

    if len(path) == 0:
        print(
            "Warning: site.getusersitepackages() returned empty string, can't determine user site packages directory"
        )
        return

    if path not in sys.path:
        print("Note: added %s to site directories" % (path,))
        site.addsitedir(path)


def maybeInstallPyMEL():
    if six.PY2 or (platform.system() != "Windows" and platform.system() != "Darwin"):
        displayPymelManualInstallInstructions()

        return False

    if not promptUserForPyMELInstall():
        return False

    mayapy = mayapyPath()

    if mayapy is None:
        displayGeneralFailure(
            "Contour Rig Tools couldn't find your Maya python installation."
        )

        return False

    pymelWhl = pymelWhlPath()

    if not os.path.exists(pymelWhl):
        displayGeneralFailure(
            "Contour Rig Tools couldn't find PyMEL installation data."
        )

        return False

    succeeded, output, canceled = installPyMEL(mayapy, pymelWhl)

    if canceled:
        displayInstallCanceled()

        return False

    if not succeeded:
        displayFailedWithOutput("PyMEL Failed to Install", output)

        return False

    ensureUserSitePackagesAdded()

    try:
        import pymel.core
    except ImportError:
        displayFailedWithOutput(
            "PyMEL Failed to Install",
            output + "\n(Install succeeded but import failed)",
        )

        return False
    except:
        err = traceback.format_exc()
        displayFailedWithOutput("PyMEL Failed to Load After Installation", err)

        return False

    displayInstallSucceeded()

    return True
